<?php

namespace uncanny_learndash_groups;

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Class GroupManagementRegistration
 * @package uncanny_learndash_groups
 */
class GroupManagementRegistration {
	static $code_details;

	/**
	 * GroupManagementRegistration constructor.
	 */
	public function __construct() {
		add_shortcode( 'uo_groups_registration_form', array( $this, 'ulgm_user_registration_func' ) );
		add_shortcode( 'uo_groups_redemption_form', array( $this, 'ulgm_code_redemption_func' ) );
		//Only fire if default registration is used
		add_action( 'init', array( $this, 'ulgm_registration_add_new_member' ), 99 );
		//Only fire if default registration is used
		add_action( 'init', array( $this, 'ulgm_registration_redeem_member' ), 99 );

		// Enqueue Scripts for uo_group_management shortcode
		add_action( 'wp_enqueue_scripts', array( $this, 'uo_group_management_registration_scripts' ) );
	}

	/**
	 * Loads all scripts and styles required by the shortcode
	 *
	 * @since 2.2.0
	 */

	public function uo_group_management_registration_scripts() {
		global $post;

		// Only add scripts if shortcode is present on page
		if ( Utilities::has_shortcode( $post, 'uo_groups_registration_form' ) || Utilities::has_block( $post, 'uncanny-learndash-groups/uo-groups-registration-form' ) ) {
			// Load styles
			wp_register_style( 'ulgm-frontend', Utilities::get_asset( 'frontend', 'bundle.min.css' ), array(), Utilities::get_version() );
			$user_colors = Utilities::user_colors();
			wp_add_inline_style( 'ulgm-frontend', $user_colors );
			wp_enqueue_style( 'ulgm-frontend', $user_colors );
		}
	}

	/**
	 * @param $atts
	 *
	 * @return string
	 */
	public function ulgm_user_registration_func( $atts ) {
		ob_start();
		//$registration_enabled = get_option( 'users_can_register' );
		$atts = shortcode_atts( array(
			'redirect'      => '',
			'code_optional' => 'no',
			'auto_login'    => 'yes',
			'role'          => 'subscriber',
			'enable_terms'  => 'yes',
		), $atts, 'ulgm_user_registration' );

		$GLOBALS['atts'] = $atts;

		// only show the registration form if allowed
		//if ( $registration_enabled ) {
		// show any error messages after form submission
		update_option( 'ulgm-custom-registration-atts', $atts );

		$this->ulgm_show_error_messages();
		include_once( Utilities::get_include( 'user-registration-form.php' ) );

		return ob_get_clean();
	}

	/**
	 * @param $atts
	 *
	 * @return string
	 */
	public function ulgm_code_redemption_func( $atts ) {
		ob_start();
		//$registration_enabled = get_option( 'users_can_register' );
		$atts = shortcode_atts( array(), $atts, 'ulgm_code_redemption' );

		$GLOBALS['atts'] = $atts;

		$this->ulgm_show_error_messages();
		include_once( Utilities::get_include( 'user-code-redemption.php' ) );

		return ob_get_clean();
	}


	/**
	 * @return \WP_Error
	 */
	public static function ulgm_registration_errors() {
		static $wp_error; // Will hold global variable safely

		return isset( $wp_error ) ? $wp_error : ( $wp_error = new \WP_Error( null, null, null ) );
	}

	/**
	 * function to catch all errors for default registration form
	 */
	public function ulgm_show_error_messages() {
		$codes = $this->ulgm_registration_errors()->get_error_codes();
		if ( $codes ) {
			echo '<div class="uncanny_learndash_codes_errors">';
			// Loop error codes and display errors
			foreach ( $codes as $code ) {
				$message = $this->ulgm_registration_errors()->get_error_message( $code );
				printf( '<span class="uo_groups_error"><strong>Error</strong>: %s </span>', esc_html( $message ) );
			}
			echo '</div>';
		}
	}

	/**
	 *
	 */
	public function ulgm_registration_add_new_member() {
		if ( isset( $_POST['_ulgm_nonce'] ) && wp_verify_nonce( $_POST['_ulgm_nonce'], Utilities::get_prefix() ) ) {
			$username          = explode( '@', $_POST['ulgm_user_email'] );
			$user_login        = sanitize_user( sanitize_title( $username[0] ), false );
			$user_email        = sanitize_email( $_POST['ulgm_user_email'] );
			$user_first        = sanitize_text_field( $_POST['ulgm_user_first'] );
			$user_last         = sanitize_text_field( $_POST['ulgm_user_last'] );
			$user_pass         = $_POST['ulgm_user_pass'];
			$pass_confirm      = $_POST['ulgm_user_pass_confirm'];
			$code_registration = $_POST['ulgm_code_registration'];
			$redirect_to       = $_POST['redirect_to'];
			$default           = array(
				'redirect'      => '',
				'code_optional' => 'no',
				'auto_login'    => 'yes',
				'role'          => 'subscriber',
			);
			$options           = get_option( 'ulgm-custom-registration-atts', $default );

			if ( username_exists( $user_login ) ) {
				// Username already registered
				$this->ulgm_registration_errors()->add( 'username_unavailable', esc_html__( 'Username already taken', 'uncanny-learndash-groups' ) );
			}
			if ( ! validate_username( $user_login ) ) {
				// invalid username
				$this->ulgm_registration_errors()->add( 'username_invalid', esc_html__( 'Invalid username', 'uncanny-learndash-groups' ) );
			}
			if ( '' === $user_login ) {
				// empty username
				$this->ulgm_registration_errors()->add( 'username_empty', esc_html__( 'Please enter a username', 'uncanny-learndash-groups' ) );
			}
			if ( ! is_email( $user_email ) ) {
				//invalid email
				$this->ulgm_registration_errors()->add( 'email_invalid', esc_html__( 'Invalid email', 'uncanny-learndash-groups' ) );
			}
			if ( email_exists( $user_email ) ) {
				//Email address already registered
				$this->ulgm_registration_errors()->add( 'email_used', esc_html__( 'Email already registered', 'uncanny-learndash-groups' ) );
			}
			if ( '' === $user_pass ) {
				// passwords do not match
				$this->ulgm_registration_errors()->add( 'password_empty', esc_html__( 'Please enter a password', 'uncanny-learndash-groups' ) );
			}
			if ( $pass_confirm !== $user_pass ) {
				// passwords do not match
				$this->ulgm_registration_errors()->add( 'password_mismatch', esc_html__( 'Passwords do not match', 'uncanny-learndash-groups' ) );
			}
			if ( '' === $code_registration && 'no' === $options['code_optional'] ) {
				$this->ulgm_registration_errors()->add( 'code_empty', esc_html__( 'Registration code is empty', 'uncanny-learndash-groups' ) );
			} elseif ( ! empty( $code_registration ) ) {
				$code_details = SharedFunctions::is_key_available( $code_registration );
				if ( is_array( $code_details ) ) {
					if ( 'failed' === $code_details['result'] ) {
						if ( 'invalid' === $code_details['error'] ) {
							$this->ulgm_registration_errors()->add( 'code_invalid', Config::$invalid_code );
						} elseif ( 'existing' === $code_details['error'] ) {
							$this->ulgm_registration_errors()->add( 'code_redeemed', Config::$already_redeemed );
						} elseif ( 'seat_not_available' === $code_details['error'] ) {
							$this->ulgm_registration_errors()->add( 'seat_not_available', Config::$seat_not_available );
						}
					} elseif ( 'success' === $code_details['result'] ) {
						self::$code_details = $code_details;
					}
				} else {
					self::$code_details = null;
					$this->ulgm_registration_errors()->add( 'code_invalid', Config::$invalid_code );
				}
			}

			$errors = $this->ulgm_registration_errors()->get_error_messages();

			// only create the user in if there are no errors
			if ( empty( $errors ) ) {
				$role        = key_exists( 'role', $options ) ? $options['role'] : 'subscriber';
				$new_user_id = wp_insert_user( array(
						'user_login'      => $user_login,
						'user_pass'       => $user_pass,
						'user_email'      => $user_email,
						'first_name'      => $user_first,
						'last_name'       => $user_last,
						'user_registered' => date( 'Y-m-d H:i:s' ),
						'role'            => $role,
					)
				);
				if ( $new_user_id ) {
					// send an email to the admin alerting them of the registration
					wp_new_user_notification( $new_user_id, null, 'admin' );
					// log the new user in
					if ( ! class_exists( 'Database' ) ) {
						include_once Utilities::get_include( 'database.php' );
					}
					if ( is_array( self::$code_details ) && 'success' === self::$code_details['result'] ) {
						update_user_meta( $new_user_id, '_ulgm_code_used', $code_registration );
						$result = SharedFunctions::set_user_to_code( $new_user_id, $code_registration, 'Not Started' );
						if ( $result ) {
							SharedFunctions::set_user_to_group( $new_user_id, self::$code_details['ld_group_id'] );
						}
					}
					if ( 'yes' === $options['auto_login'] ) {
						wp_set_auth_cookie( $new_user_id );
						wp_set_current_user( $new_user_id, $user_login );
					}

					if ( ! empty( $redirect_to ) ) {
						wp_redirect( $redirect_to . '?' . $_REQUEST['key'] . '&registered&nonce=' . wp_create_nonce( time() ) );
					} else {
						wp_redirect( get_permalink() . '?' . $_REQUEST['key'] . '&registered&nonce=' . wp_create_nonce( time() ) );
					}
					exit;
				}
			}
		}
	}

	/**
	 *
	 */
	public function ulgm_registration_redeem_member() {
		if ( isset( $_POST['_ulgm_code_nonce'] ) && wp_verify_nonce( $_POST['_ulgm_code_nonce'], Utilities::get_prefix() ) ) {
			$code_redeem = $_POST['ulgm_code_redeem'];

			if ( ! empty( $code_redeem ) && is_user_logged_in() ) {
				$code_details = SharedFunctions::is_key_available( $code_redeem );
				if ( is_array( $code_details ) ) {
					if ( 'failed' === $code_details['result'] ) {
						if ( 'invalid' === $code_details['error'] ) {
							$this->ulgm_registration_errors()->add( 'code_invalid', Config::$invalid_code );
						} elseif ( 'existing' === $code_details['error'] ) {
							$this->ulgm_registration_errors()->add( 'code_redeemed', Config::$already_redeemed );
						} elseif ( 'seat_not_available' === $code_details['error'] ) {
							$this->ulgm_registration_errors()->add( 'seat_not_available', Config::$seat_not_available );
						}
					} elseif ( 'success' === $code_details['result'] ) {
						self::$code_details = $code_details;
					}
				} else {
					self::$code_details = null;
					$this->ulgm_registration_errors()->add( 'code_invalid', Config::$invalid_code );
				}
			}

			$errors = $this->ulgm_registration_errors()->get_error_messages();
			// only create the user in if there are no errors
			if ( empty( $errors ) ) {
				$user_id = wp_get_current_user()->ID;
				if ( $user_id ) {
					// log the new user in
					if ( ! class_exists( 'Database' ) ) {
						include_once Utilities::get_include( 'database.php' );
					}
					if ( is_array( self::$code_details ) && 'success' === self::$code_details['result'] ) {
						update_user_meta( $user_id, '_ulgm_code_used', $code_redeem );
						$result = SharedFunctions::set_user_to_code( $user_id, $code_redeem, 'Not Started' );
						if ( $result ) {
							SharedFunctions::set_user_to_group( $user_id, self::$code_details['ld_group_id'] );
						}

						wp_redirect( get_permalink() . '?' . $_REQUEST['key'] . '&registered&nonce=' . wp_create_nonce( time() ) );
						exit;
					}
				}
			}
		}
	}
}