<?php
defined( 'ABSPATH' ) || die();

/**
 * Handles automatically creating ConvertKit custom fields for componenents like Lessons, Topics, and Quizzes.
 *
 * @since 1.1.0
 */
class LD_ConvertKit_ResourceTagging {

	/**
	 * LD_ConvertKit_AutoTags constructor.
	 *
	 * @since 1.1.0
	 */
	function __construct() {

		add_action( 'learndash_lesson_completed', array( $this, 'user_complete_lesson' ) );
		add_action( 'learndash_topic_completed', array( $this, 'user_complete_topic' ) );
		add_action( 'learndash_quiz_completed', array( $this, 'user_complete_quiz' ), 10, 2 );
	}

	/**
	 * Fires when a user completes a Lesson.
	 *
	 * @since 1.1.0
	 * @access private
	 *
	 * @param array $args Various arguments sent through.
	 */
	function user_complete_lesson( $args ) {

		// Check arguments
		if ( ! isset( $args['course'] ) ||
		     ! $args['course'] instanceof WP_Post ||
		     ! isset( $args['lesson'] ) ||
		     ! $args['lesson'] instanceof WP_Post ||
		     ! isset( $args['user'] ) ||
		     ! $args['user'] instanceof WP_User
		) {

			return;
		}

		$course_id = $args['course']->ID;
		$lesson_id = $args['lesson']->ID;
		$user_id   = $args['user']->ID;

		$this->subscribe_user( $lesson_id, $user_id, $course_id );
	}

	/**
	 * Fires when a user completes a Topic.
	 *
	 * @since 1.1.0
	 * @access private
	 *
	 * @param array $args Various arguments sent through.
	 */
	function user_complete_topic( $args ) {

		// Check arguments
		if ( ! isset( $args['course'] ) ||
		     ! $args['course'] instanceof WP_Post ||
		     ! isset( $args['topic'] ) ||
		     ! $args['topic'] instanceof WP_Post ||
		     ! isset( $args['user'] ) ||
		     ! $args['user'] instanceof WP_User
		) {

			return;
		}

		$course_id = $args['course']->ID;
		$topic_id  = $args['topic']->ID;
		$user_id   = $args['user']->ID;

		$this->subscribe_user( $topic_id, $user_id, $course_id );
	}

	/**
	 * Fires when a user completes a Quiz.
	 *
	 * @since 1.1.0
	 * @access private
	 *
	 * @param array $quizdata Data for the completed quiz.
	 * @param WP_User $user Current user object.
	 */
	function user_complete_quiz( $quizdata, $user ) {

		// Check arguments
		if ( ! isset( $quizdata['course'] ) ||
		     ! $quizdata['course'] instanceof WP_Post ||
		     ! isset( $quizdata['quiz'] ) ||
		     ! $quizdata['quiz'] instanceof WP_Post ||
		     ! $user instanceof WP_User
		) {

			return;
		}

		$course_id = $quizdata['course']->ID;
		$quiz_id   = $quizdata['quiz']->ID;
		$user_id   = $user->ID;

		$this->subscribe_user( $quiz_id, $user_id, $course_id );
	}

	/**
	 * Subscribes a user to a resource tag.
	 *
	 * User must already be subscribed to the Course Tag.
	 *
	 * @since 1.1.0
	 * @access private
	 *
	 * @param string|int $resource_id Resource to subscribe the user to (Course, Lesson, Topic, Quiz).
	 * @param string|int $user_ID User to subscribe.
	 * @param string|int $course_ID Course to subscribe within.
	 *
	 * @return bool False on fail, true on success.
	 */
	private function subscribe_user( $resource_id, $user_ID, $course_ID = 0 ) {

		if ( ! LD_ConvertKit_API() ) {

			return false;
		}

		$subscriber_ID = ld_convertkit_get_user_subscribe_id( $user_ID );

		// Make sure user is subscribed
		if ( ! $subscriber_ID ||
		     ! LearnDash_ConvertKit()->courses->is_user_subscribed( $user_ID, $course_ID, true )
		) {

			// User is not subscribed AND auto-subscribe is not enabled. Bail.
			if ( ld_convertkit_get_option_field( 'auto_subscribe' ) !== '1' ) {

				return false;
			}

			$subscriber_ID = LearnDash_ConvertKit()->courses->subscribe_user( $user_ID, $course_ID );
		}

		// Still can't subscribe? Whatevs... just bail
		if ( ! $subscriber_ID ) {

			return false;
		}

		$resource_tag = ld_convertkit_ensure_resource_tag( $resource_id, array(
			'name' => get_the_title( $course_ID ) . ': ' . get_the_title( $resource_id ),
		) );

		$userdata = get_userdata( $user_ID );

		LearnDash_ConvertKit()->convertkit_api->tag_subscribe( $resource_tag, array(
			'email'      => $userdata->user_email,
			'first_name' => $userdata->first_name,
		) );
	}
}