<?php
/**
 * Plugin Name: Uncanny LearnDash Groups
 * Description: Allow LearnDash Group Leaders to manage group membership and licenses for LearnDash courses from the front end
 * Version: 3.4.8
 * Author: Uncanny Owl
 * Author URI: www.uncannyowl.com
 * Plugin URI: www.uncannyowl.com/uncanny-learndash-groups/
 * Text Domain: uncanny-learndash-groups
 * Domain Path: /languages
 */

namespace uncanny_learndash_groups;

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Class InitializePlugin
 * @package uncanny_learndash_groups
 */
class InitializePlugin {

	/**
	 * The plugin name
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string
	 */
	private $plugin_name = 'Uncanny LearnDash Groups';

	/**
	 * The plugin name acronym
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string
	 */
	private $plugin_prefix = 'ulgm';

	/**
	 * The plugin version number
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string
	 */
	private $plugin_version = '3.4.8';

	/**
	 * The full path and filename
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string
	 */
	private $path_to_plugin_file = __FILE__;

	/**
	 * Allows the debugging scripts to initialize and log them in a file
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string
	 */
	private $log_debug_messages = false;

	/**
	 * The instance of the class
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      Boot
	 */
	private static $instance = null;


	/**
	 * class constructor
	 */
	private function __construct() {
		// Define version
		if ( ! defined( 'UNCANNY_GROUPS_VERSION' ) ) {
			define( 'UNCANNY_GROUPS_VERSION', $this->plugin_version );
		}

		// Add settings link on plugin page
		$uncanny_learndash_groups_plugin_basename = plugin_basename( __FILE__ );
		add_filter( 'plugin_action_links_' . $uncanny_learndash_groups_plugin_basename, array(
			__CLASS__,
			'uncanny_learndash_groups_plugin_settings_link',
		), 22 );
		// Add feature image support to LD Groups
		add_filter( 'learndash_post_args_groups', array(
			__CLASS__,
			'uo_groups_add_support',
		) );
		// Load Utilities
		$this->initialize_utilities();

		// Load Configuration
		$this->initialize_config();

		// Load the plugin files
		$this->boot_plugin();
		add_action( 'plugins_loaded', [ $this, 'uncanny_learndash_groups_text_domain' ] );

	}


	// Allow Translations to be loaded
	public function uncanny_learndash_groups_text_domain() {
		load_plugin_textdomain( 'uncanny-learndash-groups', false, basename( dirname( __FILE__ ) ) . '/languages/' );
	}

	/**
	 * @param $links
	 *
	 * @return mixed
	 */
	public static function uncanny_learndash_groups_plugin_settings_link( $links ) {
		$settings_link = '<a href="' . admin_url( 'admin.php?page=uncanny-learndash-groups-licensing' ) . '">' . __( 'Licensing', 'uncanny-learndash-groups' ) . '</a>';
		array_unshift( $links, $settings_link );
		$settings_link = '<a href="' . admin_url( 'admin.php?page=uncanny-groups' ) . '">' . __( 'Settings', 'uncanny-learndash-groups' ) . '</a>';
		array_unshift( $links, $settings_link );

		return $links;
	}

	/**
	 * Creates singleton instance of class
	 *
	 * @return InitializePlugin $instance The InitializePlugin Class
	 * @since 1.0.0
	 *
	 */
	public static function get_instance() {

		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Initialize Static singleton class that has shared function and variables that can be used anywhere in WP
	 *
	 * @since 1.0.0
	 */
	private function initialize_utilities() {

		include_once( dirname( __FILE__ ) . '/src/utilities.php' );
		Utilities::set_date_time_format();

	}

	/**
	 * Initialize Static singleton class that configures all constants, utilities variables and handles activation/deactivation
	 *
	 * @since 1.0.0
	 */
	private function initialize_config() {

		include_once( dirname( __FILE__ ) . '/src/config.php' );
		$config_instance = Config::get_instance();

		$plugin_name = apply_filters( $this->plugin_prefix . '_plugin_name', $this->plugin_name );

		$config_instance->configure_plugin_before_boot( $plugin_name, $this->plugin_prefix, $this->plugin_version, $this->path_to_plugin_file, $this->log_debug_messages );

	}

	/**
	 * Initialize Static singleton class auto loads all the files needed for the plugin to work
	 *
	 * @since 1.0.0
	 */
	private function boot_plugin() {

		include_once( dirname( __FILE__ ) . '/src/boot.php' );
		Boot::get_instance();

	}

	/**
	 * Add support to LearnDash Groups for feature image.
	 *
	 * @param array $args LD group post type.
	 *
	 * @return array
	 * @since 3.0.0
	 */
	public static function uo_groups_add_support( $args ) {
		if ( isset( $args['supports'] ) ) {
			$args['supports'][] = 'thumbnail';
		} else {
			$args['supports'] = [ 'thumbnail' ];
		}

		return $args;
	}
}


global $learndash_post_types;

if ( isset( $learndash_post_types ) ) {
	// Let's run it.
	InitializePlugin::get_instance();
} else {

	function uo_groups_learndash_not_activated() {
		?>
		<div class="notice notice-error">
			<h4>
				<?php echo __( 'Warning: Uncanny LearnDash Groups requires LearnDash. Please install LearnDash before using the plugin.', 'uncanny-learndash-groups' ); ?>
			</h4>
		</div>
		<?php
	}

	add_action( 'admin_notices', 'uncanny_learndash_groups\uo_groups_learndash_not_activated' );
}


/**
 * @param null $group_title
 * @param int $seats
 * @param array $course_ids
 * @param null $first_name
 * @param null $last_name
 * @param null $email
 */
function uo_add_new_group_details( $group_title = null, $seats = 0, $course_ids = [], $first_name = null, $last_name = null, $email = null ) {
	//Validation!
	if ( is_user_logged_in() && ! is_null( $group_title ) && 0 !== $seats && is_array( $course_ids ) && ! empty( $course_ids ) && ! is_null( $first_name ) && ! is_null( $last_name ) && ! is_null( $email ) && is_email( $email ) ) {
		$args = [
			'ulgm_group_leader_first_name' => $first_name,
			'ulgm_group_leader_last_name'  => $last_name,
			'ulgm_group_leader_email'      => $email,
			'ulgm_group_name'              => $group_title,
			'ulgm_group_total_seats'       => $seats,
			'ulgm_group_courses'           => $course_ids,
		];
		ProcessManualGroup::process( $args );
	}
}
