<?php

namespace uncanny_learndash_groups;

// If this file is called directly, abort.

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Class AdminMenu
 * @package uncanny_learndash_groups
 */
class GroupManagementInterface {


	/**
	 * An array of localized and filtered strings that are used in templates
	 *
	 * @since    1.0.0
	 * @access   static
	 * @var      array
	 */
	static $ulgm_management_shortcode = array();


	/**
	 * This group ID that is loaded going to be load for management
	 *
	 * @since    1.0.0
	 * @access   v
	 * @var      int
	 */
	static $ulgm_current_managed_group_id = false;

	/**
	 * The group objects that are managed by the user
	 *
	 * @since    1.0.0
	 * @access   static
	 * @var      array
	 */
	static $ulgm_managed_group_objects = array();

	/**
	 * An array of user with an array of data values
	 *
	 * @since    1.0.0
	 * @access   static
	 * @var      array
	 */
	static $ulgm_enrolled_users_data = array();

	/**
	 * An array of user with an array of data values
	 *
	 * @since    1.0.0
	 * @access   static
	 * @var      array
	 */
	static $ulgm_group_leaders_data = array();

	/**
	 * The locked in group leader that created the group
	 *
	 * @since    1.0.0
	 * @access   static
	 * @var      array
	 */
	static $ulgm_locked_group_leader_id = array();

	/**
	 * class constructor
	 */
	public function __construct() {

		// Enqueue Scripts for uo_group_management shortcode
		add_action( 'wp_enqueue_scripts', array( $this, 'uo_group_management_scripts' ) );

		/* ADD FILTERS ACTIONS FUNCTION */
		add_shortcode( 'uo_groups', array( $this, 'uo_group_mgr' ), 10, 2 );
		add_shortcode( 'uo_groups_url', array( $this, 'uo_groups_url' ), 10 );


	}

	/**
	 * Loads all scripts and styles required by the shortcode
	 *
	 * @since 1.0.0
	 */
	function uo_group_management_scripts() {

		global $post;

		// Only add scripts if shortcode is present on page
		if ( Utilities::has_shortcode( $post, 'uo_groups' ) || Utilities::has_block( $post, 'uncanny-learndash-groups/uo-groups' ) ) {
			// Load needed scripts
			wp_enqueue_script( 'ulgm-modernizr', Utilities::get_vendor( 'modernizr/js/modernizr.min.js' ), array( 'jquery' ), Utilities::get_version(), true );
			wp_enqueue_script( 'ulgm-jquery-lazyload', Utilities::get_vendor( 'jquery-lazyload/js/jquery.lazyload.min.js' ), array( 'jquery' ), Utilities::get_version(), true );
			wp_enqueue_script( 'ulgm-jplist-core', Utilities::get_vendor( 'jplist/js/jplist.core.min.js' ), array( 'jquery' ), Utilities::get_version(), true );
			wp_enqueue_script( 'ulgm-jplist-history-bundle', Utilities::get_vendor( 'jplist/js/jplist.history-bundle.min.js' ), array(
				'jquery',
				'ulgm-jplist-core',
			), Utilities::get_version(), true );
			wp_enqueue_script( 'ulgm-jplist-pagination-bundle', Utilities::get_vendor( 'jplist/js/jplist.pagination-bundle.min.js' ), array(
				'jquery',
				'ulgm-jplist-core',
			), Utilities::get_version(), true );
			wp_enqueue_script( 'ulgm-jplist-textbox-filter', Utilities::get_vendor( 'jplist/js/jplist.textbox-filter.min.js' ), array(
				'jquery',
				'ulgm-jplist-core',
			), Utilities::get_version(), true );
			wp_enqueue_script( 'ulgm-jplist-sort-buttons', Utilities::get_vendor( 'jplist/js/jplist.sort-buttons.min.js' ), array(
				'jquery',
				'ulgm-jplist-core',
			), Utilities::get_version(), true );
			wp_enqueue_style( 'ulgm-font-awesome', Utilities::get_vendor( 'fontawesome/css/font-awesome.min.css' ), array(), Utilities::get_version() );

			// Setup group management JS with localized WP Rest API variables @see rest-api-end-points.php
			wp_register_script( 'ulgm-frontend', Utilities::get_asset( 'frontend', 'bundle.min.js' ), array(
				'jquery',
			), Utilities::get_version(), true );

			// API data
			$api_setup = array(
				'root'  => esc_url_raw( rest_url() . 'ulgm_management/v1/' ),
				'nonce' => \wp_create_nonce( 'wp_rest' ),
			);

			wp_localize_script( 'ulgm-frontend', 'ulgmRestApiSetup', $api_setup );
			wp_enqueue_script( 'ulgm-frontend' );

			wp_register_style( 'ulgm-frontend', Utilities::get_asset( 'frontend', 'bundle.min.css' ), array(), Utilities::get_version() );
			$user_colors = Utilities::user_colors();
			wp_add_inline_style( 'ulgm-frontend', $user_colors );
			wp_enqueue_style( 'ulgm-frontend', $user_colors );

			wp_enqueue_editor();

			// Load File saving utility
			wp_enqueue_script( 'ulgm-file-saver', Utilities::get_vendor( 'filesaver/js/filesaver.js' ), array(), Utilities::get_version(), true ); // @see https://github.com/eligrey/FileSaver.js

			// Load Modal
			wp_enqueue_script( 'ulgm-modal', Utilities::get_vendor( 'jquery-modal/js/jquery.modal.js' ), array( 'jquery' ), Utilities::get_version(), true ); // @see https://raw.githubusercontent.com/kylefox/jquery-modal/master/jquery.modal.js
			wp_enqueue_script( 'ulgm-select2', Utilities::get_vendor( 'select2/js/select2.min.js' ), array( 'jquery' ), Utilities::get_version(), true );
			wp_enqueue_style( 'ulgm-modal', Utilities::get_vendor( 'jquery-modal/css/jquery.modal.css' ), array(), Utilities::get_version() ); // @see https://raw.githubusercontent.com/kylefox/jquery-modal/master/jquery.modal.css
			wp_enqueue_style( 'ulgm-select2', Utilities::get_vendor( 'select2/css/select2.min.css' ), array(), Utilities::get_version() );
		}
	}

	/**
	 * @param $atts
	 *
	 * @return string
	 */
	function uo_groups_url( $atts ) {
		if ( is_user_logged_in() ) {
			$atts = shortcode_atts( array(
				'text' => __( 'Group Management', 'uncanny-learndash-groups' ),
			), $atts, 'uo_groups_url' );

			$user = wp_get_current_user();
			if ( in_array( 'group_leader', $user->roles ) || in_array( 'administrator', $user->roles ) ) {
				$url = SharedFunctions::get_group_management_page_id( true );

				return sprintf( '<a href="%s" class="button anchor btn btn-large uo-btn">%s</a>', $url, $atts['text'] );
			}
		}

		return '';
	}

	/**
	 * Return the HTML template that is displayed by the shortcode
	 *
	 * @param array $attributes The attributes passed in the the shortcode
	 * @param string $content The content contained by the shortcode
	 *
	 * @return string $shortcode_template The HTML template loaded
	 * @since 1.0.0
	 *
	 */
	function uo_group_mgr( $attributes, $content = '' ) {

		global $post;
		//if ( $post->ID !== SharedFunctions::get_group_management_page_id() ) {
		//	return;
		//}
		ob_start();
		/// make sure that the users has groups

		$validated = $this->set_group_id();

		if ( ! $validated['success'] ) {
			return $validated['message'];
		}

		$this->localize_filter_globalize_text();

		$this->set_enrolled_users_data();

		$this->set_group_leaders_data();

		/*
		* Show & hide sections
		*/
		$attributes = shortcode_atts( array(
			'group_name_selector'               => 'show',
			'add_courses_button'                => 'show',
			'seats_quantity'                    => 'show',
			'add_seats_button'                  => 'show',
			'add_user_button'                   => 'show',
			'remove_user_button'                => 'show',
			'upload_users_button'               => 'show',
			'download_keys_button'              => 'show',
			'progress_report_button'            => 'show',
			'quiz_report_button'                => 'show',
			'assignments_button'                => 'show',
			'essays_button'                     => 'show',
			'key_column'                        => 'show',
			'group_leader_section'              => 'show',
			'add_group_leader_button'           => 'show',
			'key_options'                       => 'show',
			'group_email_button'                => 'hide',
			'progress_management_report_button' => 'show'
		), $attributes, 'uo_groups' );

		// General
		$group_name_selector = Utilities::show_section( $attributes['group_name_selector'] );

		// Group Courses
		$add_courses_button = Utilities::show_section( $attributes['add_courses_button'] );

		// Enrolled Users @ title
		$seats_quantity   = Utilities::show_section( $attributes['seats_quantity'] );
		$add_seats_button = Utilities::show_section( $attributes['add_seats_button'] );

		// Enrolled Users @ buttons
		$add_user_button    = Utilities::show_section( $attributes['add_user_button'] );
		$remove_user_button = Utilities::show_section( $attributes['remove_user_button'] );

		// Group Email
		$add_group_email_button = Utilities::show_section( $attributes['group_email_button'] );
		$user                   = wp_get_current_user();

		$upload_users_button               = Utilities::show_section( $attributes['upload_users_button'] );
		$download_keys_button              = Utilities::show_section( $attributes['download_keys_button'] );
		$progress_report_button            = Utilities::show_section( $attributes['progress_report_button'] );
		$progress_management_report_button = Utilities::show_section( $attributes['progress_management_report_button'] );
		$quiz_report_button                = Utilities::show_section( $attributes['quiz_report_button'] );
		$assignment_button                 = Utilities::show_section( $attributes['assignments_button'] );
		$essay_button                      = Utilities::show_section( $attributes['essays_button'] );

		$key_column = Utilities::show_section( $attributes['key_column'] );

		$key_options = Utilities::show_section( $attributes['key_options'] );

		// Groups Leaders
		$add_group_leader_button = Utilities::show_section( $attributes['add_group_leader_button'] );
		$group_leader_section    = Utilities::show_section( $attributes['group_leader_section'] );

		/*
		* Include template
		*/

		include( Utilities::get_template( 'frontend-uo_groups.php' ) );

		return ob_get_clean();

	}

	/**
	 * Set the group id that the management page is to load
	 *
	 * @return array
	 * @since 1.0.0
	 *
	 */
	function set_group_id() {

		$validation = array(
			'success' => false,
			'message' => '',
		);

		$group_leader_id = get_current_user_id();

		// Allow admin to override group_leader_id
		if ( current_user_can( 'manage_options' ) && isset( $_GET['user_id'] ) ) {
			$group_leader_id = absint( $_GET['user_id'] );
		}

		// Validate the user as a group leader
		if ( ! current_user_can( 'manage_options' ) ) {
			if ( ! learndash_is_group_leader_user( $group_leader_id ) ) {
				$validation['message'] = __( 'You do not have permission to manage groups.', 'uncanny-learndash-groups' );

				return $validation;
			}
		}


		$user_group_ids = learndash_get_administrators_group_ids( $group_leader_id, true );

		if ( empty( $user_group_ids ) ) {
			$validation['message'] = __( 'You do not have any groups available to manage.', 'uncanny-learndash-groups' );

			return $validation;
		}

		$args = array(
			'numberposts' => 9999,
			'include'     => array_map( 'intval', $user_group_ids ),
			'post_type'   => 'groups',
			//'meta_key'    => SharedFunctions::$code_group_id_meta_key,
			'orderby'     => 'title',
			'order'       => 'ASC'
		);

		self::$ulgm_managed_group_objects = get_posts( $args );

		if ( ! empty( self::$ulgm_managed_group_objects ) ) {
			foreach ( self::$ulgm_managed_group_objects as $key => $group_object ) {
				$total_seats = SharedFunctions::total_seats( $group_object->ID );
				if ( 0 === $total_seats ) {
					unset( self::$ulgm_managed_group_objects[ $key ] );
				}
			}
			if ( isset( self::$ulgm_managed_group_objects[0]->ID ) ) {
				self::$ulgm_current_managed_group_id = self::$ulgm_managed_group_objects[0]->ID;
			}


		} else {

			$validation['message'] = __( 'You do not have any groups available to manage.', 'uncanny-learndash-groups' );

			return $validation;
		}

		// Maybe override current group id
		if ( isset( $_GET['group-id'] ) ) {

			$current_group_id = $_GET['group-id'];
			if ( ! in_array( $current_group_id, $user_group_ids ) ) {
				$validation['message'] = __( 'You do not have permission to manage this group.', 'uncanny-learndash-groups' );

				return $validation;
			}
			self::$ulgm_current_managed_group_id = $current_group_id;

		}

		$validation['success'] = true;

		return $validation;

	}

	/**
	 * Get and globalize user data
	 *
	 * @since 1.0.0
	 */
	function set_enrolled_users_data() {

		// Set user that are in the system
		$groups_user_object = learndash_get_groups_users( self::$ulgm_current_managed_group_id );

		// Set leaders that are in the system
		$groups_group_object = learndash_get_groups_administrator_ids( self::$ulgm_current_managed_group_id );

		// Set group enrolled courses
		$learndash_group_enrolled_courses = learndash_group_enrolled_courses( self::$ulgm_current_managed_group_id );

		// Set progress link
		$user_progress_link = GroupManagementInterface::$ulgm_management_shortcode['text']['group_progress_link'];

		// Set progress management link
		$user_progress_management_link = GroupManagementInterface::$ulgm_management_shortcode['text']['group_progress_link'];

		global $wpdb;

		// Completion
		$q_completions = "
							SELECT post_id as course_id, user_id, activity_completed
							FROM {$wpdb->prefix}learndash_user_activity
							WHERE activity_type = 'course'
							AND activity_completed IS NOT NULL
							AND activity_completed <> 0
							";

		$completions = $wpdb->get_results( $q_completions );

		$completions_rearranged = [];

		foreach ( $completions as $completion ) {

			$completions_rearranged[ (int) $completion->user_id ][ (int) $completion->course_id ] = true;
		}

		// In-progress
		$q_in_progress = "
						SELECT a.post_id as course_id, user_id
						FROM {$wpdb->prefix}learndash_user_activity a
						WHERE a.activity_type = 'course'
						AND a.activity_completed = 0
						AND ( a.activity_started != 0 || a.activity_updated != 0)
						";

		$in_progress = $wpdb->get_results( $q_in_progress );

		$in_progress_rearranged = [];

		foreach ( $in_progress as $progress ) {
			$in_progress_rearranged[ (int) $progress->user_id ][ (int) $progress->course_id ] = true;
		}

		foreach ( $groups_user_object as $user ) {


			$f     = get_user_meta( $user->ID, 'first_name', true );
			$l     = get_user_meta( $user->ID, 'last_name', true );
			$email = $user->user_email;

			// Default is not completed
			$completed = false;

			// Default progress
			$in_progress = false;

			$not_started = false;

			$status = false;

			// Check group progress courses
			foreach ( $learndash_group_enrolled_courses as $course_id ) {

				if ( isset( $in_progress_rearranged[ (int) $user->ID ][ (int) $course_id ] ) ) {
					$in_progress = true;
				} else if ( isset( $completions_rearranged[ (int) $user->ID ][ (int) $course_id ] ) && false === $in_progress ) {
					$completed = true;
				} else {
					$not_started = true;
				}

			}

			// Set Status
			if ( $in_progress && ! $not_started && ! $completed ) {
				$status = '<a href="' . $user_progress_link . '?user_id=' . $user->ID . '">' . __( 'In Progress', 'uncanny-learndash-groups' ) . '</a>';
			}

			if ( $not_started && $in_progress ) {
				$status = '<a href="' . $user_progress_link . '?user_id=' . $user->ID . '">' . __( 'In Progress', 'uncanny-learndash-groups' ) . '</a>';
			}

			if ( $not_started && $completed ) {
				$status = '<a href="' . $user_progress_link . '?user_id=' . $user->ID . '">' . __( 'In Progress', 'uncanny-learndash-groups' ) . '</a>';
			}

			if ( $in_progress && $completed ) {
				$status = '<a href="' . $user_progress_link . '?user_id=' . $user->ID . '">' . __( 'In Progress', 'uncanny-learndash-groups' ) . '</a>';
			}

			if ( $completed && ! $not_started && ! $in_progress ) {
				$status = '<a href="' . $user_progress_link . '?user_id=' . $user->ID . '">' . __( 'Completed', 'uncanny-learndash-groups' ) . '</a>';
			}

			$key = SharedFunctions::get_user_code( $user->ID, self::$ulgm_current_managed_group_id );

			if ( in_array( $user->ID, $groups_group_object ) ) {
				$key = '';
			}
			if ( false === $status ) {
				$status = __( 'Not Started', 'uncanny-learndash-groups' );
			}

			self::$ulgm_enrolled_users_data[ $user->ID ] = array(
				$f,
				$l,
				$email,
				$status,
				$key,

			);
		}

		// Set user that have been sent a code but have not redeemed it
		global $wpdb;

		$order_id = SharedFunctions::get_order_id_from_group_id( self::$ulgm_current_managed_group_id );


		if ( $order_id ) {

			$admin_created_group = get_post_meta( self::$ulgm_current_managed_group_id, '_ulgm_is_custom_group_created', true );

			if ( 'yes' === $admin_created_group ) {
				$codes_group_id = $wpdb->get_var(
					$wpdb->prepare(
						'SELECT ID FROM ' . $wpdb->prefix . SharedFunctions::$db_group_tbl . ' WHERE order_id = %d', $order_id
					)
				);
				//$codes_group_id = $order_id;
			} else {
				$codes_group_id = get_post_meta( $order_id, SharedFunctions::$code_group_id_meta_key, true );
			}

			$q               = 'SELECT * FROM ' . $wpdb->prefix . SharedFunctions::$db_group_codes_tbl . ' WHERE  group_id = %d AND code_status LIKE %s AND student_id IS NULL';
			$temp_users_code = $wpdb->get_results( $wpdb->prepare( $q, $codes_group_id, 'not redeemed' ) );

			foreach ( $temp_users_code as $user ) {

				$f      = '';
				$l      = '';
				$email  = $user->user_email;
				$status = __( 'Not Enrolled', 'uncanny-learndash-groups' );
				$key    = $user->code;

				self::$ulgm_enrolled_users_data[ $key ] = array(
					$f,
					$l,
					$email,
					$status,
					$key,
				);

			}


		}
	}


	/**
	 * Get and globalize group leader data
	 *
	 * @since 1.0.0
	 */
	function set_group_leaders_data() {

		$groups_user_object = learndash_get_groups_administrators( self::$ulgm_current_managed_group_id );

		self::$ulgm_locked_group_leader_id = get_post_meta( self::$ulgm_current_managed_group_id, 'locked_admin_group_leader' );

		foreach ( $groups_user_object as $user ) {
			$f     = get_user_meta( $user->ID, 'first_name', true );
			$l     = get_user_meta( $user->ID, 'last_name', true );
			$email = $user->user_email;

			self::$ulgm_group_leaders_data[ $user->ID ] = array(

				$f,
				$l,
				$email

			);
		}

	}

	/**
	 * Filter and localize all text, then set as global for use in template file
	 *
	 * @since 1.0.0
	 */
	function localize_filter_globalize_text() {

		// Get current manged group's post object (id has been validated)
		$group_post = get_post( (int) self::$ulgm_current_managed_group_id );

		// Page heading
		$total_seats     = SharedFunctions::total_seats( $group_post->ID );
		$remaining_seats = SharedFunctions::remaining_seats( $group_post->ID );
		$available       = SharedFunctions::available_seats( $group_post->ID );
		$enrolled_seats  = $total_seats - $remaining_seats;

		$remaining_seat_text = get_option( 'ulgm_per_seat_text_plural', 'Seats' );
		$seat_text           = get_option( 'ulgm_per_seat_text_plural', 'Seats' );

		$seats_label_singular = get_option( 'ulgm_per_seat_text', 'Seat' );
		$user_label_singular  = get_option( 'ulgm_per_user_text', 'User' );
		$seats_label_plural   = get_option( 'ulgm_per_seat_text_plural', 'Seats' );
		$users_label_plural   = get_option( 'ulgm_per_user_text_plural', 'Users' );

		if ( 1 === $total_seats ) {
			$total_seats_string = sprintf( _x( '1 Total %1$s', '%1$s is the "seat" label', 'uncanny-learndash-groups' ), $seats_label_singular );
		} else {
			$total_seats_string = sprintf( _x( '%1$s Total %2$s', '%1$s is a number, %2$s is the "seats" label', 'uncanny-learndash-groups' ), $total_seats, $seats_label_plural );
		}

		if ( 1 === $remaining_seats ) {
			$remaining_seats_string = sprintf( _x( '1 %1$s Remaining', '%1$s is the "seat" label', 'uncanny-learndash-groups' ), $seats_label_singular );
		} else {
			$remaining_seats_string = sprintf( _x( '%1$s %2$s Remaining', '%1$s is a number, %2$s is the "seats" label', 'uncanny-learndash-groups' ), $remaining_seats, $seats_label_plural );
		}

		if ( 1 === $available ) {
			$available_seats_string = sprintf( _x( '1 %1$s Available', '%1$s is the "seat" label', 'uncanny-learndash-groups' ), $seats_label_singular );
		} else {
			$available_seats_string = sprintf( _x( '%1$s %2$s Available', '%1$s is a number, %2$s is the "seats" label', 'uncanny-learndash-groups' ), $available, $seats_label_plural );
		}
		
		if ( 1 === $enrolled_seats ) {
			$enrolled_seats_string = sprintf( _x( '1 %1$s', '%1$s is the "user" label', 'uncanny-learndash-groups' ), $user_label_singular );
		} else {
			$enrolled_seats_string = sprintf( _x( '%1$s %2$s', '%1$s is a number, %2$s is the "users" label', 'uncanny-learndash-groups' ), $enrolled_seats, $users_label_plural );
		}

		self::$ulgm_management_shortcode['text']['group_title']           = $group_post->post_title;
		self::$ulgm_management_shortcode['text']['group_id_select_label'] = __( 'Select Group ID', 'uncanny-learndash-groups' );

		self::$ulgm_management_shortcode['text']['x_total_seats']     = $total_seats_string;
		self::$ulgm_management_shortcode['text']['x_seats_remaining'] = $remaining_seats_string;
		self::$ulgm_management_shortcode['text']['x_seats_available'] = $available_seats_string;
		self::$ulgm_management_shortcode['text']['x_users_enrolled']  = $enrolled_seats_string;

		//Enrolled Users
		self::$ulgm_management_shortcode['text']['enrolled_users'] = __( 'Enrolled Users', 'uncanny-learndash-groups' );


		// Enrolled user table headers
		self::$ulgm_management_shortcode['table']['enrolled_users']['headers'][0] = array(
			'title' => __( 'First Name', 'uncanny-learndash-groups' ),
			'slug'  => 'first-name'
		);
		self::$ulgm_management_shortcode['table']['enrolled_users']['headers'][1] = array(
			'title' => __( 'Last Name', 'uncanny-learndash-groups' ),
			'slug'  => 'last-name'
		);
		self::$ulgm_management_shortcode['table']['enrolled_users']['headers'][2] = array(
			'title' => __( 'Email', 'uncanny-learndash-groups' ),
			'slug'  => 'email'
		);
		self::$ulgm_management_shortcode['table']['enrolled_users']['headers'][3] = array(
			'title' => __( 'Status', 'uncanny-learndash-groups' ),
			'slug'  => 'status'
		);
		self::$ulgm_management_shortcode['table']['enrolled_users']['headers'][4] = array(
			'title' => __( 'Key', 'uncanny-learndash-groups' ),
			'slug'  => 'key'
		);

		// Enrolled user buttons
		self::$ulgm_management_shortcode['text']['add_user']                = __( 'Add User', 'uncanny-learndash-groups' );
		self::$ulgm_management_shortcode['text']['upload_users']            = __( 'Upload Users', 'uncanny-learndash-groups' );
		self::$ulgm_management_shortcode['text']['email_users']             = __( 'Email Users', 'uncanny-learndash-groups' );
		self::$ulgm_management_shortcode['text']['manage_progress_button']  = __( 'Manage Progress', 'uncanny-learndash-groups' );
		self::$ulgm_management_shortcode['text']['modal_x_seats_remaining'] = sprintf( _x( 'You have %1$s %2$s remaining', '%1$s is a number, %2$s is the "seats" label', 'uncanny-learndash-groups' ), SharedFunctions::remaining_seats( $group_post->ID ), strtolower( get_option( 'ulgm_per_seat_text_plural', 'seats' ) ) );
		self::$ulgm_management_shortcode['text']['add_invite_users']        = __( 'Add and invite users', 'uncanny-learndash-groups' );
		self::$ulgm_management_shortcode['text']['add_invite_user']         = __( 'Add and invite user', 'uncanny-learndash-groups' );
		self::$ulgm_management_shortcode['text']['send_enrollment_key']     = __( 'Send enrollment key', 'uncanny-learndash-groups' );
		self::$ulgm_management_shortcode['text']['add_existing_user']       = __( 'Add existing user', 'uncanny-learndash-groups' );
		self::$ulgm_management_shortcode['text']['send_enrollment_keys']    = __( 'Send enrollment keys', 'uncanny-learndash-groups' );
		self::$ulgm_management_shortcode['text']['do_not_send_emails']      = __( 'Suppress email (keys to be manually distributed)', 'uncanny-learndash-groups' );

		self::$ulgm_management_shortcode['text']['group_progress_link'] = SharedFunctions::get_group_report_page_id( true );
		self::$ulgm_management_shortcode['text']['group_progress']      = sprintf( __( '%s Report', 'uncanny-learndash-groups' ), \LearnDash_Custom_Label::get_label( 'course' ) );

		self::$ulgm_management_shortcode['text']['group_progress_management_link'] = SharedFunctions::get_group_manage_progress_report_page_id( true );
		self::$ulgm_management_shortcode['text']['group_progress_management']      = __( 'Progress', 'uncanny-learndash-groups' );

		self::$ulgm_management_shortcode['text']['group_quiz_progress_link'] = SharedFunctions::get_group_quiz_report_page_id( true );
		self::$ulgm_management_shortcode['text']['group_quiz_progress']      = sprintf( __( '%s Report', 'uncanny-learndash-groups' ), \LearnDash_Custom_Label::get_label( 'quiz' ) );

		self::$ulgm_management_shortcode['text']['group_assignment_link'] = SharedFunctions::get_group_assignment_report_page_id( true );
		self::$ulgm_management_shortcode['text']['group_assignment_page'] = __( 'Assignments', 'uncanny-learndash-groups' );
		self::$ulgm_management_shortcode['text']['group_essay_link']      = SharedFunctions::get_group_essay_report_page_id( true );
		self::$ulgm_management_shortcode['text']['group_essay_page']      = __( 'Essays', 'uncanny-learndash-groups' );

		if ( Utilities::if_woocommerce_active() ) {
			//self::$ulgm_management_shortcode['text']['add_seats_link']   = SharedFunctions::add_group_seats_link( $group_post->ID, SharedFunctions::total_seats( $group_post->ID ) + 1 );
			self::$ulgm_management_shortcode['text']['add_seats_link']   = SharedFunctions::add_group_seats_link( $group_post->ID, 1 );
			self::$ulgm_management_shortcode['text']['add_seats']        = __( 'Add Seats', 'uncanny-learndash-groups' );
			self::$ulgm_management_shortcode['text']['buy_courses_link'] = SharedFunctions::add_buy_courses_link( $group_post->ID );
		}
		self::$ulgm_management_shortcode['text']['buy_courses']         = sprintf( __( 'Add %s', 'uncanny-learndash-groups' ), \LearnDash_Custom_Label::get_label( 'courses' ) );
		self::$ulgm_management_shortcode['text']['administration_link'] = admin_url();
		self::$ulgm_management_shortcode['text']['administration']      = __( 'Administration', 'uncanny-learndash-groups' );
		self::$ulgm_management_shortcode['text']['remove_users']        = __( 'Remove User(s)', 'uncanny-learndash-groups' );
		self::$ulgm_management_shortcode['text']['resend_invitation']   = __( 'Resend Invitation', 'uncanny-learndash-groups' );


		// Groups leader headers
		self::$ulgm_management_shortcode['text']['group_leaders'] = __( 'Group Leaders', 'uncanny-learndash-groups' );

		// Group Leaders buttons
		self::$ulgm_management_shortcode['text']['add_leader']          = __( 'Add Group Leader', 'uncanny-learndash-groups' );
		self::$ulgm_management_shortcode['text']['remove_group_leader'] = __( 'Remove Group Leader(s)', 'uncanny-learndash-groups' );

		// Group Leaders table headers
		self::$ulgm_management_shortcode['table']['group_leaders']['headers'][0] = array(
			'title' => __( 'First Name', 'uncanny-learndash-groups' ),
			'slug'  => 'first-name'
		);
		self::$ulgm_management_shortcode['table']['group_leaders']['headers'][1] = array(
			'title' => __( 'Last Name', 'uncanny-learndash-groups' ),
			'slug'  => 'last-name'
		);
		self::$ulgm_management_shortcode['table']['group_leaders']['headers'][2] = array(
			'title' => __( 'Email', 'uncanny-learndash-groups' ),
			'slug'  => 'email'
		);

		// Group Leaders buttons
		self::$ulgm_management_shortcode['text']['add_group_leader'] = __( 'Add Group Leader', 'uncanny-learndash-groups' );


		// Message is set by a redirect in the JS file
		if ( isset( $_GET['message'] ) ) {
			self::$ulgm_management_shortcode['message'] = __( $_GET['message'], 'uncanny-learndash-groups' );
		} else {
			self::$ulgm_management_shortcode['message'] = '';
		}

		self::$ulgm_management_shortcode = apply_filters( 'ulgm_management_shortcode', self::$ulgm_management_shortcode );

	}

}