<?php
/**
 * The Admin Side LearnDash ConvertKit
 *
 * @since 1.0.0
 */

defined( 'ABSPATH' ) || die();

/**
 * Class LD_ConvertKit_Admin
 *
 * The Admin Side LearnDash ConvertKit
 *
 * @since 1.0.0
 */
class LD_ConvertKit_Admin {

	/**
	 * LD_ConvertKit_Admin constructor.
	 *
	 * @since 1.0.0
	 */
	function __construct() {

		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'admin_menu', array( $this, 'admin_menu' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'learndash_admin_tabs_set', array( $this, 'admin_tabs' ) );
		add_filter( 'parent_file', array( $this, 'fix_parent_file' ) );
		add_filter( 'submenu_file', array( $this, 'fix_submenu_file' ), 10, 2 );

		if ( isset( $_GET['ld_convertkit_resync_subscribers'] ) ) {

			add_action( 'admin_init', array( $this, 'resync_subscribers' ) );
		}
	}

	/**
	 * Re-sync all subscribers.
	 *
	 * @since 1.1.0
	 * @access private
	 */
	function resync_subscribers() {

		global $wp_settings_errors;

		$result = ld_convertkit_sync_subscribers( true );

		if ( $result === true ) {

			add_settings_error(
				'ld_convertkit_sync_subscribers',
				'',
				__( 'Subscribers have been successfully re-sync\'d.', 'learndash-convertkit' ),
				'updated'
			);

		} else {

			add_settings_error(
				'ld_convertkit_sync_subscribers',
				'',
				__( 'There was an issue re-syncing subscribers. Please double check your API information and try again later.', 'learndash-convertkit' ),
				'error'
			);
		}

		set_transient( 'settings_errors', $wp_settings_errors, 30 );

		wp_redirect(
			add_query_arg( 'settings-updated', 1, admin_url( 'options-general.php?page=learndash_convertkit' ) )
		);
		exit();
	}

	/**
	 * Enqueues admin scripts.
	 *
	 * @since 1.0.0
	 * @access private
	 */
	function enqueue_scripts() {

		wp_enqueue_style( 'learndash-convertkit-admin' );
		wp_enqueue_script( 'learndash-convertkit-admin' );
	}

	/**
	 * Creates a (temporary) Submenu Item for our Admin Page
	 *
	 * @since 1.0.0
	 * @access private
	 */
	function admin_menu() {

		global $submenu;

		// We're hooking into options-general.php so that we have better access to WP's Settings API
		$page_hook = add_submenu_page(
			'options-general.php',
			_x( 'ConvertKit', 'ConvertKit Tab Label', 'learndash-convertkit' ),
			_x( 'ConvertKit', 'ConvertKit Tab Label', 'learndash-convertkit' ),
			'manage_options',
			'learndash_convertkit',
			array( $this, 'admin_page' )
		);

		add_action( "load-{$page_hook}", array( $this, 'settings_page_actions' ) );

		// Ensure that we don't have errors from a non-existing Index for non-Admins
		if ( isset( $submenu['options-general.php'] ) ) {

			$settings_index = null;

			foreach ( $submenu['options-general.php'] as $key => $menu_item ) {

				// Index 2 is always the child page slug
				if ( $menu_item[2] == 'learndash_convertkit' ) {

					$settings_index = $key;
					break;
				}
			}

			// Unset from the Submenu
			unset( $submenu['options-general.php'][ $settings_index ] );
		}
	}

	/**
	 * Fires on the Settings page load.
	 *
	 * @since 1.0.0
	 * @access private
	 */
	function settings_page_actions() {

		add_filter( 'rbm_fieldhelpers_load_select2', '__return_true' );

		if ( ld_convertkit_course_auto_sync_enabled() ) {

			$this->sync_courses();
		}
	}

	/**
	 * Syncs Courses to ConvertKit.
	 *
	 * @since 1.0.0
	 * @access private
	 */
	private function sync_courses() {

		$subscriber_form = ld_convertkit_get_option_field( 'subscriber_form' );

		if ( ! $subscriber_form ) {

			return;
		}

		$courses = get_posts( array(
			'post_type'   => 'sfwd-courses',
			'numberposts' => - 1,
		) );

		foreach ( $courses as $course ) {

			$tag = ld_convertkit_ensure_resource_tag( $course->ID );

			if ( $tag === false ) {

				add_settings_error(
					'ld_convertkit_course_sync',
					'',
					__( 'There was an error syncing Courses to ConvertKit. Please double check your API information and try again later.', 'learndash-convertkit' ),
					'error'
				);
				break;
			}
		}
	}

	/**
	 * Regsiter settings for each Field
	 *
	 * @since 1.0.0
	 * @access private
	 */
	function register_settings() {

		register_setting( 'learndash_convertkit', 'ld_convertkit_api_key' );
		register_setting( 'learndash_convertkit', 'ld_convertkit_api_secret' );
		register_setting( 'learndash_convertkit', 'ld_convertkit_subscriber_form' );
		register_setting( 'learndash_convertkit', 'ld_convertkit_auto_subscribe' );
		register_setting( 'learndash_convertkit', 'ld_convertkit_disable_course_sync' );

		add_settings_section(
			'general',
			null,
			null,
			'learndash_convertkit'
		);

		$this->add_settings_fields();
	}

	/**
	 * Adds all settings fields.
	 *
	 * @since 1.0.0
	 * @access private
	 */
	private function add_settings_fields() {

		$settings = array();

		$settings['learndash_convertkit'] = array();

		// API Key
		$settings['learndash_convertkit'][] = array(
			'section'  => 'general',
			'id'       => 'api_key',
			'label'    => __( 'API Key', 'learndash-convertkit' ),
			'callback' => 'ld_convertkit_do_field_text',
			'args'     => array(
				'input_class' => 'regular-text',
			),
		);

		// API Secret
		$settings['learndash_convertkit'][] = array(
			'section'  => 'general',
			'id'       => 'api_secret',
			'label'    => __( 'API Secret', 'learndash-convertkit' ),
			'callback' => 'ld_convertkit_do_field_text',
			'args'     => array(
				'input_class' => 'regular-text',
			),
		);

		if ( LD_ConvertKit_API() !== false ) {

			$forms = LD_ConvertKit_API()->get_forms();

			if ( $forms === 0 ) {

				// Subscriber Form Error
				$settings['learndash_convertkit'][] = array(
					'section'  => 'general',
					'id'       => 'subscriber_form_error',
					'label'    => __( 'Subscribe Students To Form', 'learndash-convertkit' ),
					'callback' => 'ld_convertkit_do_field_html',
					'args'     => array(
						'html' => __( 'Error in getting forms from ConvertKit. Please double check your API key and secret are valid or try again later.', 'learndash-convertkit' ),
					),
				);

			} else {

				$form_options = array();

				foreach ( $forms as $form ) {

					$form_options[] = array(
						'text'  => $form['name'],
						'value' => $form['id'],
					);
				}

				// Subscriber Form
				$settings['learndash_convertkit'][] = array(
					'section'  => 'general',
					'id'       => 'subscriber_form',
					'label'    => __( 'Subscribe Users To Form', 'learndash-convertkit' ),
					'callback' => 'ld_convertkit_do_field_select',
					'args'     => array(
						'no_options_text' => sprintf(
						/* translators: %s are wrapping ConvertKit in a link */
							__( 'No forms available. Please create a new form in %sConvertKit%s.', 'learndash-convertkit' ),
							'<a href="https://app.convertkit.com/landing_pages/new" target="_blank">',
							'</a>'
						),
						'options'         => $form_options,
						'input_class'     => 'regular-text',
						'placeholder'     => __( '- Select a Form -', 'learndash-convertkit' ),
						'select2_options' => array(
							'allowClear' => true,
						),
					),
				);
			}
		}

		// Auto Subscribe
		$settings['learndash_convertkit'][] = array(
			'section'  => 'general',
			'id'       => 'auto_subscribe',
			'label'    => __( 'Auto Subscribe', 'learndash-convertkit' ) . ld_convertkit_get_field_tip(
					__( 'If enabled, users will be automatically subscribed to ConvertKit and added to the Course tag when they enroll. If disabled, users must manually subscribe.', 'learndash-convertkit' )
				),
			'callback' => 'ld_convertkit_do_field_toggle',
		);

		// Sync Course Tags
		$settings['learndash_convertkit'][] = array(
			'section'  => 'general',
			'id'       => 'disable_course_sync',
			'label'    => __( 'Auto Sync Course Tags', 'learndash-convertkit' ) . ld_convertkit_get_field_tip(
					__( 'If enabled, ConvertKit Tags will be automatically created for each Course. If disabled, Tags will only be created when a user subscribes to a Course.', 'learndash-convertkit' )
				),
			'callback' => 'ld_convertkit_do_field_toggle',
			'args'     => array(
				'unchecked_value' => '1',
				'checked_value'   => '0',
				'default'         => '0',
			),
		);

		// Re-Sync Subscribers
		$settings['learndash_convertkit'][] = array(
			'section'  => 'general',
			'id'       => 'resync_subscribers',
			'label'    => __( 'Re-Sync Subscribers', 'learndash-convertkit' ) . ld_convertkit_get_field_tip(
					__( 'Erase all cached subscription data and sync it again. Do this if you have made extreme changes in ConvertKit and those changes are not reflected in the user base on this website. WARNING: It is recommended to backup this website\'s database before doing this because lost data is non-recoverable.', 'learndash-convertkit' )
				),
			'callback' => 'ld_convertkit_do_field_html',
			'args'     => array(
				'html' => '<a href="' . add_query_arg( 'ld_convertkit_resync_subscribers', '1' ) . '" class="button">' . __( 'Do it, I understand the risks', 'learndash-convertkit' ) . '</a>',
			),
		);

		/**
		 * All plugin settings fields displayed on the setting page.
		 *
		 * @since 1.0.0
		 */
		$settings = apply_filters( 'ld_convertkit_settings_fields', $settings );

		foreach ( $settings as $page => $page_settings ) {

			foreach ( $page_settings as $setting ) {

				add_settings_field(
					$setting['id'],
					$setting['label'],
					$setting['callback'],
					$page,
					$setting['section'],
					array_merge(
						array(
							'option_field' => true,
							'name'         => $setting['id'],
						),
						isset( $setting['args'] ) ? $setting['args'] : array()
					)
				);
			}
		}
	}

	/**
	 * Output our Admin Page (Finally!)
	 *
	 * @since 1.0.0
	 * @access private
	 */
	function admin_page() {

		settings_errors();
		?>

        <div class="wrap">

            <h2><?php _e( 'LearnDash ConvertKit Settings', 'learndash-convertkit' ); ?></h2>

            <form method="post" action="options.php">
				<?php settings_fields( 'learndash_convertkit' ); ?>

				<?php do_settings_sections( 'learndash_convertkit' ); ?>

				<?php submit_button(); ?>
            </form>

        </div>
		<?php
	}

	/**
	 * Fix the Admin Title since our pages "don't exist"
	 *
	 * @since 1.0.0
	 * @access private
	 *
	 * @param string $admin_title The page title, with extra context added
	 * @param string $title The original page title
	 *
	 * @return string Admin Title
	 */
	function admin_title( $admin_title, $title ) {

		global $current_screen;

		if ( $current_screen->base == 'settings_page_learndash_convertkit' ) {

			return __( 'LearnDash ConvertKit Settings', 'learndash-convertkit' ) . $admin_title;
		}

		return $admin_title;
	}

	/**
	 * Adds the Admin Tab using LD v2.4's new method
	 *
	 * @since 1.0.0
	 * @access private
	 *
	 * @param string $admin_menu_section Admin Menu Section
	 */
	function admin_tabs( $admin_menu_section = '' ) {

		if ( $admin_menu_section == 'admin.php?page=learndash_lms_settings' ) {

			learndash_add_admin_tab_item(
				'admin.php?page=learndash_lms_settings',
				array(
					'id'   => 'settings_page_learndash_convertkit',
					'link' => add_query_arg( array( 'page' => 'learndash_convertkit' ), 'options-general.php' ),
					'name' => _x( 'ConvertKit', 'ConvertKit Tab Label', 'learndash-convertkit' ),
				),
				40
			);
		}
	}

	/**
	 * Fakes the Current Menu Item
	 *
	 * @since 1.0.0
	 * @access private
	 *
	 * @param string $parent_file Parent Menu Item
	 *
	 * @return string Modified String
	 */
	function fix_parent_file( $parent_file ) {

		global $current_screen, $self;

		if ( $current_screen->base == 'settings_page_learndash_convertkit' ) {

			// Render this as the Active Page Menu
			$parent_file = 'admin.php?page=learndash_lms_settings';

			// Ensure the top-level "Settings" doesn't show as active
			$self = 'learndash-lms';
		}

		return $parent_file;
	}

	/**
	 * Fakes the current Submenu Item
	 *
	 * @since 1.0.0
	 * @access private
	 *
	 * @param string $submenu_file Current Menu Item
	 * @param string $parent_file Parent Menu Item
	 *
	 * @return string Modified String
	 */
	function fix_submenu_file( $submenu_file, $parent_file ) {

		global $current_screen;

		if ( $current_screen->base == 'settings_page_learndash_convertkit' ) {

			$submenu_file = 'admin.php?page=learndash_lms_settings';
		}

		return $submenu_file;
	}
}